﻿"""
A transport which "jumps-to" pre-specified destinations.
"""

import os
import pickle

import viz
import vizact
import vizinput
import vizshape

from transportation import teleport


class JumpTo(teleport.Teleport):
	"""Teleport which also provides location management and indexing.
	
	@param locationFilename: path of file containing a location list.
	
	@param targetMarker: node serving as visual representation of the
	currently selected jump-to location (To which the transport will 
	travel upon calling the `jumpTo` method).
	
	@param accelerationTime: Deprecated, has no effect.
	
	@param useOrientation: Deprecated, please use base class parameter
	`rotationMode` instead.
	
	@param forcePivotToOrientation: Deprecated, please use base class
	parameter `rotationMode` instead.
	
	@param forcePivotToPosition: Deprecated, please use base class
	parameter `translationMode` instead.
	"""
	
	LOCK_JUMP_TO = 1
	
	def __init__(self,
					locationFilename='',
					targetMarker=None,
					accelerationTime=None,  # deprecated
					useOrientation=None,  # deprecated
					forcePivotToOrientation=None,  # deprecated
					forcePivotToPosition=None,  # deprecated
					preventDuplicateLocations=False,
					**kwargs):
		
		self._preventDuplicates = preventDuplicateLocations
		
		# Handle deprecated keyword arguments
		if accelerationTime is not None:
			viz.logWarn('**Warning: accelerationTime is deprecated and has no effect.')
		
		# Cast deprecated useOrientation into rotationMode
		if useOrientation is not None:
			viz.logWarn('**Warning: useOrientation is deprecated. Use parameter rotationMode.')
			if useOrientation:
				kwargs['rotationMode'] = teleport.MATCH_TRANSPORT
			else:
				kwargs['rotationMode'] = teleport.DISABLED
		
		# Cast deprecated forcePivotToOrientaion into rotationMode
		if forcePivotToOrientation is not None:
			viz.logWarn('**Warning: forcePivotToOrientation is deprecated. Use parameter rotationMode.')
			if useOrientation and forcePivotToOrientation:
				kwargs['rotationMode'] = teleport.MATCH_PIVOT
			else:
				kwargs['rotationMode'] = teleport.DISABLED
		
		# Cast deprecated forcePivotToPosition into positionMode
		if forcePivotToPosition is not None:
			viz.logWarn('**Warning: forcePivotToPosition is deprecated. Use parameter positionMode.')
			if forcePivotToPosition:
				kwargs['positionMode'] = teleport.MATCH_PIVOT
			else:
				kwargs['positionMode'] = teleport.DISABLED
		
		# Init the base class
		super(JumpTo, self).__init__(**kwargs)
		
		# Store passed-in parameters
		self._locationFilename = locationFilename
		self._targetMarker = targetMarker
		
		# Maintain collection of jump-to locations
		self._currentIndex = 0
		self._locationList = []
		
		# A numerical threshold for comparing location coordinates
		self._matchThreshold = 0.0001
		
		# Load a location file, if supplied
		if self._locationFilename:
			self.load(self._locationFilename)
	
	def createConfigUI(self):
		"""Creates the vizconfig ui (inherited from superclass).
		
		No need to call this function directly.
		"""
		ui = super(JumpTo, self).createConfigUI()
		ui.addCommandItem('Next', 'Next', self.selectNext)
		ui.addCommandItem('Prev', 'Prev', self.selectPrev)
		ui.addCommandItem('Jump To', 'Jump To', self.jumpTo)
		return ui
	
	def addLocation(self, location=None):
		"""Appends the current location to the maintained collection.
		
		@param location: a 2-tuple of the form (pos, ori)
		where pos is of the form [x, y, z], and ori is of the form
		[yaw, pitch, roll]
		"""
		if not self._frameLocked or not self._deferred:
			# Use the transports current location
			if location is None:
				location = self._getLocation()
			# Use the passed-in location, if it's not already stored
			if not self._preventDuplicates or not self._hasLocation(location):
				self._locationList.append(location)
				self.selectIndex(len(self._locationList)-1)
		self._lockRequested |= self.LOCK_JUMP_TO
	
	def clear(self):
		"""Empties the current location list"""
		self._locationList = []
		if self._targetMarker:
			self._targetMarker.visible(True)
	
	def jumpTo(self, index=None):
		"""Jumps to the location at index via teleport
		
		@param index: the index of a location.
		"""
		# don't do anything if we have any empty location list
		if not self._locationList:
			return
		
		# if an index is specified, select it first
		if index is not None:
			self.selectIndex(index)
		
		# perform the teleport operation
		if not self._frameLocked or not self._deferred:
			targetPos, targetEuler = self._locationList[self._currentIndex]
			super(JumpTo, self).teleportTo(targetPos, targetEuler)
		
		self._lockRequested |= self.LOCK_JUMP_TO
	
	def load(self, filename=""):
		"""Loads a location list from a give file. 
		
		File should contain a list of 2-tuples of the form (pos, ori)
		where pos is of the form [x, y, z], and ori is of the form
		[yaw, pitch, roll].
		"""
		if not filename:
			filename = self._getFilename()
		
		# Load the location data from the given file from disk
		tupleList = []
		if filename:
			try:
				with open(filename, 'r') as tempFile:
					if os.path.splitext(filename)[1] == '.p':
						tupleList = pickle.load(tempFile)
					else:
						for line in tempFile.readlines():
							tupleList.append(eval(line))
			except IOError as e:
				viz.logWarn("*** Warning: unable to load location database. {}", e)
		self.setLocationList(tupleList)
		
		self.selectIndex(self._currentIndex)
	
	def removeLocation(self, location=None):
		"""Removes the current location from the list of locations.
		
		@param location: a 2-tuple of the form (pos, ori)
		where pos is of the form [x, y, z], and ori is of the form
		[yaw, pitch, roll]
		"""
		index = self._getLocationIndex(location)
		if index != -1:
			self._locationList.pop(index)
	
	def save(self):
		"""Writes the stored location list data to the attributed file.
		
		If no attributed file is stored with this object, a warning is
		given, and nothing else is performed.
		"""
		try:
			# Save the locations to the stored file
			if self._locationFilename:
				with open(self._locationFilename, 'w') as tempFile:
					if os.path.splitext(self._locationFilename)[1] == '.p':
						pickle.dump(self._locationList, tempFile)
					else:
						for location in self._locationList:
							tempFile.write(str(location)+"\n")
		except IOError:
			viz.logWarn("*** Warning: unable to write location database.")
	
	def selectNext(self):
		"""Sets selection to the next location in the collection"""
		if len(self._locationList) > 0:
			self.selectIndex(self._currentIndex+1)
	
	def selectPrev(self):
		"""Sets selection to previous location in the collection"""
		if len(self._locationList) > 0:
			self.selectIndex(self._currentIndex-1)
	
	def selectIndex(self, index):
		"""Sets selection to location at `index` in the collection.
		
		@param index: Int: index to be set as selection.
		"""
		if len(self._locationList) > 0:
			if not self._frameLocked or not self._deferred:
				self._currentIndex = index % len(self._locationList)
				self._onSelection()
			self._lockRequested |= self.LOCK_JUMP_TO
		elif self._targetMarker:
			self._targetMarker.visible(True)
	
	def setLocationList(self, list):
		"""Sets the internally maintained collection of locations.
		
		@param list: [] a list of 2-tuples of the form (pos, ori)
		where pos is of the form [x, y, z], and ori is of the form
		[yaw, pitch, roll]
		"""
		self._locationList = list[:]
		self.selectIndex(len(self._locationList)-1)
	
	def _getFilename(self):
		"""Opens a vizard fileopen dialog to retrieve a filename"""
		return vizinput.fileOpen(directory='./')
	
	def _getLocation(self):
		"""Returns a 2-tuple with pos, euler of current location"""
		accuracy = 3  # numerical precision
		
		if self._pivot == None:
			# Sample the location of the self
			pos = self.getPosition(viz.ABS_GLOBAL)
			euler = self.getEuler(viz.ABS_GLOBAL)
		else:
			# Sample the position of the pivot itself
			pos = self._pivot.getPosition(viz.ABS_GLOBAL)
			euler = self._pivot.getEuler(viz.ABS_GLOBAL)
		
		# Round the coordinates to fewer digits
		pos[0] = round(pos[0], accuracy)
		pos[1] = round(pos[1], accuracy)
		pos[2] = round(pos[2], accuracy)
		euler[0] = round(euler[0], accuracy)
		euler[1] = round(euler[1], accuracy)
		euler[2] = round(euler[2], accuracy)
		
		# Return the rounded location coordinates
		return (pos, euler)
	
	def _getLocationIndex(self, location):
		"""Checks whether `location` is already in the collection.
		
		Applies a numerical check based on stored threshold value.
		
		@param location: a 2-tuple of the form (pos, ori)
		where pos is of the form [x, y, z], and ori is of the form
		[yaw, pitch, roll]
		
		@return Int, the index of the provided `location` in the
		internally maintained collection. -1 if no match.
		"""
		if self._locationList == []:
			return -1
		for locIndex in range(0, len(self._locationList)):
			pos, euler = self._locationList[locIndex]
			newPos, newEuler = location
			match = True
			for i in range(0, 3):
				if abs(pos[i] - newPos[i]) > self._matchThreshold or abs(euler[i] != newEuler[i]) > self._matchThreshold:
					match = False
			if match:
				return locIndex
		return -1
	
	def _hasLocation(self, location):
		"""Checks if the location exists in the stored collection.
		
		@param location: a 2-tuple of the form (pos, ori)
		where pos is of the form [x, y, z], and ori is of the form
		[yaw, pitch, roll]
		
		@return Bool
		"""
		index = self._getLocationIndex(location)
		if index == -1:
			return False
		return True
	
	def _onSelection(self):
		"""Callback triggered when a location has been selected"""
		if self._targetMarker:
			# Place the targetMarker visible at the selected location
			pos, euler = self._locationList[self._currentIndex]
			self._targetMarker.setPosition(pos, viz.ABS_GLOBAL)
			self._targetMarker.setEuler(euler, viz.ABS_GLOBAL)
			self._targetMarker.setText(str(pos))


class TargetMarker(viz.VizNode):
	"""A node representing a target location in the environment.
	
	Designed to be used by a jump to transport.
	Can be used as is or subclassed in order to provide a more
	sophisticated visual representation of a targeted location.
	
	@param size: the radius of the node geometry in meters.
	"""
	def __init__(self, size=0.1):
		# Drawable node representing "shown" state
		self._shown = vizshape.addSphere(size)
		self._shown.visible(False)
		
		# Init the base class with the "shown" node
		viz.VizNode.__init__(self, id=self._shown.id)
		
		# Drawable node representing "hidden" state
		self._hidden = vizshape.addSphere(size)
		self._hidden.alpha(0.5)
		self._hidden.disable(viz.DEPTH_TEST)
		self._hidden.drawOrder(10000)
		self._hidden.setParent(self._shown)
		
		# Drawable text node providing annotation
		self._text = viz.addText('location 1')
		self._text.alignment(viz.ALIGN_LEFT_TOP)
		self._text.fontSize(60)
		self._text.setBackdrop(viz.BACKDROP_OUTLINE)
		self._text.billboard(viz.BILLBOARD_VIEW)
		self._text.setAutoScale(True)
		self._text.drawOrder(10002)
		self._text.disable(viz.DEPTH_TEST)
		self._text.visible(False)
		self._text.setParent(self._shown)
	
	def remove(self, *args, **kwargs):
		"""Removes the current marker, and related resources"""
		super(TargetMarker, self).remove(*args, **kwargs)
		self._shown.remove()
		self._hidden.remove()
		self._text.remove()
	
	def select(self):
		"""Changes visual appearance when marker is selected"""
		self._shown.color(0, 1, 0)
		self._hidden.color(0, 1, 0)
	
	def setText(self, text):
		"""Sets the text for the target marker"""
		self._text.message(text)
	
	def unselect(self):
		"""Changes visual appearance when marker is unselected"""
		self._shown.color(1, 1, 1)
		self._hidden.color(1, 1, 1)


if __name__ == "__main__":
	import vizconfig
	
	viz.go()
	
	# Load an environment model
	piazza = viz.add('piazza.osgb')
	
	# Set up a convenient starting view
	viz.MainView.setPosition(0, 1, -4)
	
	# Initialize a target marker and transport
	marker = TargetMarker()
	marker.visible(True)
	transportRepresentation = viz.add("beachball.osgb")
	jumpTo = JumpTo(node=transportRepresentation, targetMarker=marker)
	
	# Set up a location list for the jump-to
	jumpTo.setLocationList([([1, 0, 0], [0, 0, 0]),
							([0, 2, 0], [0, 0, 0]),
							([0, 0, 1], [0, 0, 0]),
							])
	
	# Define keyboard controls to use the transport
	vizact.onkeydown(viz.KEY_UP, jumpTo.selectNext)
	vizact.onkeydown(viz.KEY_DOWN, jumpTo.selectPrev)
	vizact.onkeydown(' ', jumpTo.jumpTo)
	vizact.onkeydown('0', jumpTo.clear)
	vizact.onkeydown('1', jumpTo.addLocation, ([0, 0, 0], [90, 0, 0]))
	
	# Get a handle to the transport's config window and make visible
	name = jumpTo.getConfigName()
	vizconfig.getConfigWindow(name).setWindowVisible(True)
